var polygonThis;
function PolygonEditor(canvasSelector){
    this.canvas = document.querySelector(canvasSelector);
    this.ctx = this.canvas.getContext('2d');
    //polygons多边形数组
    this.polygons = [];
    this.startX = null;
    this.startY = null;
    this.rectTouch = false;
    //是否可编辑
    this.editable = false;
    //是否可拖拽
    this.draggable = false;
    //多边形填充颜色
    this.polygonFillColor = 'rgba(122,255,66,0.5)';
    //多边形线条颜色
    this.polygonStrokeStyle = '#474b4f';
    //多边形线条宽度
    this.polygonLineWidth = 2;
    //引脚填充颜色
    this.rectFillColor = '#474b4f';
    //引脚边框颜色
    this.rectStrokeStyle = '#474b4f';
    //引脚边框宽度
    this.rectLineWidth = 1;
    //多边形闪烁颜色
    this.flashTask = null;
    this.enableFlash = false;
    this.flashColor = ['rgba(255, 0, 0, 0.65)','rgba(255, 153, 153, 0.65)'];
    this.tempFlashColor = null;
    //记录初始宽度,用于计算实际缩放比例
    this.initWidth = this.canvas.width;
    this.init();
    polygonThis = this;
}

/**
* 鼠标按下事件
*/
function polygonMouseDownEvent(e){
    var trigger = false;
    const x = e.offsetX, y = e.offsetY;
    polygonThis.polygons.forEach(function(polygon, i) {
        polygonThis.polygons[i].rects = polygonThis.rectForEach(polygonThis.polygons[i].rects, x, y, true);
        const rect = polygonThis.polygons[i].rects.filter(function(r){return r.down})[0];
        //鼠标左键按下拖动引脚或者多边形
        if(e.button == 0){
            //引脚移动
            if (rect) {
                trigger = true;
            }
            //判断鼠标按下选中了多边形
            if (polygonThis.pointInPolygon(x, y, polygonThis.polygons[i].points)) {
                //把当前多边形移动到数组最后的位置
                const p = polygonThis.polygons[i];
                polygonThis.polygons.splice(i, 1);
                polygonThis.polygons.push(p);
                //移动开启
                polygonThis.polygons[polygonThis.polygons.length - 1].polygonTouch = true;
                //记录移动前的坐标
                polygonThis.startX = x;
                polygonThis.startY = y;
                trigger = true;
            }
        }else if(e.button == 2){
            //重置引脚down状态
            if(rect){
                polygonThis.polygons[i].rects[rect.index].down = false;
            }
            //鼠标右键销毁多边形的实点, 实点至少要保留3个,维持最少边的多边形(三角形)
            if(rect && polygonThis.polygons[i].points.length > 6 && polygonThis.polygons[i].points[rect.index].solid){
                trigger = true;
                //先计算出需要生成的新虚点
                var prePoint,nextPoint;
                if(rect.index - 2 < 0){
                    prePoint = polygonThis.polygons[i].points[polygonThis.polygons[i].points.length -2];
                }else{
                    prePoint = polygonThis.polygons[i].points[rect.index - 2];
                }
                if(rect.index + 2 > polygonThis.polygons[i].points.length - 1){
                    nextPoint = polygonThis.polygons[i].points[0];
                }else{
                    nextPoint = polygonThis.polygons[i].points[rect.index + 2];
                }
                var newPoint = {
                    x:(prePoint.x + nextPoint.x)/2.0,
                    y:(prePoint.y + nextPoint.y)/2.0,
                    solid:false
                }
                //移除3个点,添加1个虚点
                if(rect.index - 1 >= 0){
                    polygonThis.polygons[i].points.splice(rect.index-1, 3, newPoint);
                }else{
                    polygonThis.polygons[i].points.splice(-1, 1, newPoint);
                    polygonThis.polygons[i].points.splice(rect.index, 2);
                }
                //图形重绘, 并更新引脚坐标数据
                polygonThis.polygons[i].rects = polygonThis.repaint()[i];
            }
        }

    });
    if(trigger){
        polygonThis.rectTouch = true;
        e.stopPropagation();
    }
}

/**
* 鼠标移动事件
*/
function polygonMouseMoveEvent(e){
    var trigger = false;
    const x = e.offsetX , y = e.offsetY;

    polygonThis.polygons.forEach(function(polygon, i) {
      polygonThis.polygons[i].rects = polygonThis.rectForEach(polygonThis.polygons[i].rects, x, y, false);

      const rect = polygonThis.polygons[i].rects.filter(function(r){return r.touch})[0];
      //引脚移动
      if (rect) {
        polygonThis.polygons[i].points[rect.index].x = x;
        polygonThis.polygons[i].points[rect.index].y = y;
        if(polygonThis.polygons[i].points[rect.index].solid){
            var prePointIndex,nextPointIndex,preSolidPoint,netSolidPoint;
            if(rect.index - 2 < 0){
                preSolidPoint = polygonThis.polygons[i].points[polygonThis.polygons[i].points.length - 2];
                prePointIndex = polygonThis.polygons[i].points.length - 1;
            }else{
                preSolidPoint = polygonThis.polygons[i].points[rect.index - 2];
                prePointIndex = rect.index - 1;
            }
            if(rect.index + 2  >= polygonThis.polygons[i].points.length){
                netSolidPoint = polygonThis.polygons[i].points[0];
                nextPointIndex = polygonThis.polygons[i].points.length - 1;
            }else{
                netSolidPoint = polygonThis.polygons[i].points[rect.index + 2];
                nextPointIndex = rect.index + 1;
            }
            //更新前后虚点
            polygonThis.polygons[i].points[prePointIndex].x = (preSolidPoint.x + x)/2.0;
            polygonThis.polygons[i].points[prePointIndex].y = (preSolidPoint.y + y)/2.0;;
            polygonThis.polygons[i].points[nextPointIndex].x = (netSolidPoint.x + x)/2.0;
            polygonThis.polygons[i].points[nextPointIndex].y = (netSolidPoint.y + y)/2.0;
        }
        //图形重绘
        polygonThis.repaint();
        trigger = true;
      }
      //多边形整体移动
      if (polygonThis.polygons[i].polygonTouch && !rect) {
        polygonThis.polygons[i].points.forEach(function(point) {
          point.x += x - polygonThis.startX;
          point.y += y - polygonThis.startY;
        });
        //图形重绘
        polygonThis.repaint();
        polygonThis.startX = x;
        polygonThis.startY = y;
        trigger = true;
      }
    });
    if(trigger || polygonThis.rectTouch){
        e.stopPropagation();
    }
}

/**
* 鼠标松开事件
*/
function polygonMouseUpEvent(e){
     polygonThis.rectTouch = false;
     var trigger = false;
     const x = e.offsetX, y = e.offsetY;

    polygonThis.polygons.forEach(function(polygon, i) {
        const rect = polygonThis.polygons[i].rects.filter(function(r){return r.touch})[0];
       //多边形移动结束
        if (polygonThis.polygons[i].polygonTouch && !rect) {
            polygonThis.polygons[i].points.forEach(function(point) {
                point.x += x - polygonThis.startX;
                point.y += y - polygonThis.startY;
            });
            //图形重绘, 并更新引脚坐标数据
            polygonThis.polygons[i].rects = polygonThis.repaint()[i];
            trigger = true;
       }

       polygonThis.polygons[i].polygonTouch = false;
       //引脚移动结束
        if (rect) {
            polygonThis.polygons[i].rects[rect.index].touch = false;
            polygonThis.polygons[i].rects[rect.index].down = false;
            polygonThis.polygons[i].points[rect.index].x = x;
            polygonThis.polygons[i].points[rect.index].y = y;
            //被拖动的虚点会变成实点,并产生2个新的虚点
            if(!polygonThis.polygons[i].points[rect.index].solid){
                polygonThis.polygons[i].points[rect.index].solid = true;
                var prePoint = {
                    x:(polygonThis.polygons[i].points[rect.index - 1].x + x)/2.0,
                    y:(polygonThis.polygons[i].points[rect.index - 1].y + y)/2.0,
                    solid:false
                }
                var nextIndex = rect.index + 1;
                if(rect.index + 1 >= polygonThis.polygons[i].points.length){
                    nextIndex = 0;
                }
                var nextPoint = {
                    x:(polygonThis.polygons[i].points[nextIndex].x + x)/2.0,
                    y:(polygonThis.polygons[i].points[nextIndex].y + y)/2.0,
                    solid:false
                }
                var currentPoint = polygonThis.polygons[i].points[rect.index];
                polygonThis.polygons[i].points.splice(rect.index, 1, prePoint, currentPoint, nextPoint);
            }
         //图形重绘, 并更新引脚坐标数据
         polygonThis.polygons[i].rects = polygonThis.repaint()[i];
         trigger = true;
       }
     });
     if(trigger){
         e.stopPropagation();
     }
}

PolygonEditor.prototype = {
    constructor:PolygonEditor,
    /**判断点击的坐标(x,y)在多边形内
    * @param x 点击的横坐标
    * @param y 点击的纵坐标
    * @param points 多边形点位集合
    */
    pointInPolygon: function(x, y, points){
      var i, j = points.length - 1, touch = false;
      for (i = 0; i < points.length; i++) {
        const pxi = points[i].x,
          pxj = points[j].x,
          pyi = points[i].y,
          pyj = points[j].y;

        if (
          ((pyi < y && pyj >= y) || (pyj < y && pyi >= y)) && (pxi <= x || pxj <= x)
        ) {
          if (pxi + (y - pyi) / (pyj - pyi) * (pxj - pxi) < x) {
            touch = !touch;
          }
        }
        j = i;
      }
      return touch;
    },

    /**
    * 计算多边形点位坐标
    * @param k 多边形的边数
    * @param x 多边形横坐标left x
    * @param y 多边形纵坐标top y
    */
    polygonPoints : function(k, x, y) {
        const points = [];
        var lastPoint;
        for (var i = 0; i < k; i++) {
            //正多边形点坐标计算公式
            var p = {
              x: 75 * Math.sin(Math.PI * (i * (360 / k) / 180)) + x,
              y: 75 * Math.cos(Math.PI * (i * (360 / k) / 180)) + y,
              solid:true //实点
            };
            //计算添加虚点
            if(lastPoint == null){
                lastPoint = p;
            }else{
                points.push({
                    x:(lastPoint.x + p.x)/2.0,
                    y:(lastPoint.y + p.y)/2.0,
                    solid:false
                });
                lastPoint = p;
            }
            //添加实点
            points.push(p);
        }
        //最后一个虚点需要另外计算,因为最后线闭合是靠closePath自动闭合的
        points.push({x:(lastPoint.x + points[0].x)/2.0, y:(lastPoint.y + points[0].y)/2.0, solid:false});
        return points;
    },

    /**
    * 绘制多边形
    */
    drawPolygon : function(polygons) {
		_this = this;
        _this.ctx.clearRect(0, 0, _this.canvas.width, _this.canvas.height);
        //多边形引脚集合
        const rectss = [];
        if(_this.enableFlash){
            _this.tempFlashColor = _this.flashColor[0] == _this.tempFlashColor ? _this.flashColor[1] : _this.flashColor[0];
        }
        polygons.forEach(function(polygon){
            const rects = [];
            var minX,maxX,minY,maxY;
            _this.ctx.save();
            _this.ctx.beginPath();
            polygon.points.forEach(function(p, i) {
              _this.ctx.lineTo(p.x, p.y);
            });
            if(polygon.enableFlash){
                _this.ctx.fillStyle = _this.tempFlashColor;
            }else{
                _this.ctx.fillStyle = polygon.fillColor || _this.polygonFillColor;
            }
            _this.ctx.fill();
            _this.ctx.lineWidth =  _this.polygonLineWidth;
            _this.ctx.strokeStyle = _this.polygonStrokeStyle;
            _this.ctx.closePath();
            _this.ctx.stroke();
            _this.ctx.restore();
            //绘制多边形点的引脚
            polygon.points.forEach(function(p, i) {
                if (i < polygon.points.length) {
                    //可编辑开启时,才绘制引脚
                    if(_this.editable){
                        _this.ctx.save();
                        _this.ctx.translate(p.x, p.y);
                        _this.ctx.beginPath();
                        _this.ctx.rect(-5, -5, 10, 10);
                        _this.ctx.strokeStyle = _this.rectStrokeStyle;
                        _this.ctx.lineWidth = _this.rectLineWidth;
                        _this.ctx.stroke();
                        if(p.solid){
                            _this.ctx.fillStyle = _this.rectFillColor;
                            _this.ctx.fill();
                        }
                        _this.ctx.restore();
                    }
                    rects.push({
                      x: p.x,
                      y: p.y,
                      w: 10,
                      h: 10,
                      index: i,
                    });
                    //计算点位xy最值
                    minX = minX == null || p.x < minX ? p.x : minX;
                    maxX = maxX == null || p.x > maxX ? p.x : maxX;
                    minY = minY == null || p.y < minY ? p.y : minY;
                    maxY = maxY == null || p.y > maxY ? p.y : maxY;
                }
            });
            rectss.push(rects);
            //绘制文字
            _this.ctx.font = "35px Arial";
            _this.ctx.textAlign = "center";
            _this.ctx.fillText(polygon.data.entityName, (minX + maxX)/2, (minY + maxY)/2+10);
        });
        return rectss;
    },

    /**
    * 更新多边形的引脚数据,判断是否被鼠标选中、拖动
    */
    rectForEach : function(rects, x, y, down) {
        x += 5;
        y += 5;
        rects.forEach(function(rect, i) {
            //鼠标按下时
            if (down) {
              //判断按下的点坐标是否和多边形的引脚相交
              if (
                x >= rect.x &&
                x <= rect.x + rect.w &&
                y >= rect.y &&
                y <= rect.y + rect.h
              ) {
                //标记被按下的引脚
                rect.down = true;
              }
            } else {
              //鼠标移动中判断鼠标坐标和多边形相交,且引脚是按下状态, 更新引脚touch状态true,用于过滤筛选
              if (
                x >= rect.x &&
                x <= rect.x + rect.w &&
                y >= rect.y &&
                y <= rect.y + rect.h &&
                rect.down
              ) {
                rect.touch = true;
              }
            }
            });
        //返回更新后的引脚数据
        return rects;
    },

    /**
    * 指定坐标添加正多边形
    * @param n 多边形边数
    * @param x 横坐标
    * @param y 纵坐标
	* @param id 多边形唯一标识
    * @param data 自定义数据
    */
    add : function(n, x, y, id, data){
        var checkExist =  polygonThis.polygons.filter(function(r) {return r.id == id})[0];
        if(!checkExist){
            var pointsData = this.polygonPoints(n, x, y);
            var pointsArray = this.polygons.map(function(p) {return p.points});
            pointsArray.push(pointsData);
            this.polygons.push({
                id: data.entityId,
                points: pointsData,//点位集合
                rects: null,//引脚
                enableFlash: false,
                polygonTouch: false,
                data:data
            });
            this.polygons[pointsArray.length - 1].rects = this.drawPolygon(this.polygons)[pointsArray.length - 1];

        }
    },

    /**
    * 载入多边形
    */
    loadPolygonsData : function(polygonsData){
        if(polygonsData != null && polygonsData.length > 0){
            this.polygons = polygonsData;
            //计算虚点
            this.polygons.forEach(function(polygon, i){
                var lastPoint;
                var newPoints = [];
                polygon.points.forEach(function(p, j) {
                    //计算添加虚点
                    if(lastPoint == null){
                        lastPoint = p;
                    }else{
                        newPoints.push({
                            x:(lastPoint.x + p.x)/2.0,
                            y:(lastPoint.y + p.y)/2.0,
                            solid:false
                        });
                        lastPoint = p;
                    }
                    //添加实点
                    newPoints.push(p);
                })
                //最后一个虚点需要另外计算
                newPoints.push({x:(lastPoint.x + polygon.points[0].x)/2.0, y:(lastPoint.y + polygon.points[0].y)/2.0, solid:false});
                polygon.points = newPoints;
            });
            //绘制更新引脚
            var rects = this.drawPolygon(this.polygons);
            this.polygons.forEach(function(polygon, i) {
                 polygon.rects = rects[i];
             });
        }
    },

    /*
    * 获取多边形数据
    */
    getPolygonsData : function(){
        var scale = Number(this.canvas.width/this.initWidth);
        return this.polygons.map(function(polygon) {
            var pointsArray = [];
            polygon.points.forEach(function(point, i){
                if(point.solid){
                    pointsArray.push({
                        positionX:(point.x / scale),
                        positionY:(point.y / scale),
                        sortNo:i+1
                    })
                }

            })
            return {
             id: polygon.data.mapElementId,
             mapId:polygon.data.mapId,
             entityType:polygon.data.entityType,
             entityId:polygon.data.entityId,
             points:pointsArray
            }
        });
    },

    /**
    * 开启多边形编辑
    */
    enableEdit : function(){
        //显示引脚
        this.editable = true;
        this.repaint();
        //开启拖拽
        if(!this.draggable){
            //监听鼠标按下事件
            this.canvas.addEventListener('mousedown', polygonMouseDownEvent);
            //监听鼠标移动
            this.canvas.addEventListener('mousemove', polygonMouseMoveEvent);
            //监听鼠标放开事件
            this.canvas.addEventListener('mouseup', polygonMouseUpEvent);
            this.draggable = true;
        }
    },

    /**
    * 关闭多边形编辑
    */
    disableEdit : function(){
        //隐藏引脚
        this.editable = false;
        this.repaint();
        //关闭拖拽
        if(this.draggable){
            //监听鼠标按下事件
            this.canvas.removeEventListener('mousedown', polygonMouseDownEvent);
            //监听鼠标移动
            this.canvas.removeEventListener('mousemove', polygonMouseMoveEvent);
            //监听鼠标放开事件
            this.canvas.removeEventListener('mouseup', polygonMouseUpEvent);
            this.draggable = false;
        }
    },

    /**
    * 更新多边形闪烁状态
    * entityIds 开启闪烁的多边形ids,不在指定ids内的其他多边形都会被关闭闪烁
    */
    updateFlash :function(entityIds){
        var _this = this;
        var enableCount = 0;
        this.polygons.forEach(function(polygon) {
            polygon.enableFlash = entityIds != null && entityIds.length > 0
                && entityIds.indexOf(polygon.data.entityId) != -1;
            if(polygon.enableFlash){
                _this.enableFlash = true;
                enableCount ++;
            }
        });
        if(enableCount == 0){
            window.clearInterval(_this.flashTask);
            _this.flashTask = null;
            _this.enableFlash = false;
            _this.repaint();
        }else if(_this.flashTask == null){
            _this.flashTask = window.setInterval(function(){
                _this.repaint();
            },500);
        }

    },

    /**
    * 选中指定的多边形, 用颜色闪动的效果表示被选中的多边形
    */
    choose: function(entityId){
        var _this = this;
        this.polygons.forEach(function(polygon, index){
            if(entityId != null && polygon.id == entityId){
                var showhandle = window.setInterval(function(){
                    polygon.fillColor = polygon.fillColor != "rgba(0,0,255,0.5)"?
                        "rgba(0,0,255,0.5)" : _this.polygonFillColor;
                    _this.repaint();
                }, 300)
                window.setTimeout(function(){
                    window.clearInterval(showhandle);
                    polygon.fillColor = _this.polygonFillColor
                    _this.repaint();
                }, 2000);
            }else{
                polygon.fillColor = null;
            }
        })
        this.repaint();
    },

    /**
    * 返回指定entityId多边形是否存在
    */
    ifExist: function(entityId){
        return this.polygons.filter(function(p){return p.id == entityId})[0] != null;
    },

    /**
    * 比例缩放重绘
    * @param scale 缩放比例
    */
    repaint : function(scale){
        if(scale != null && scale > 0){
            //画布缩放
            this.canvas.width = scale * this.canvas.width;
            this.canvas.height = scale * this.canvas.height;
            //多边形缩放计算
            this.polygons.forEach(function(polygon, i) {
                polygon.points.forEach(function(point, j) {
                    point.x = scale * point.x;
                    point.y = scale * point.y;
                });
            });
        }
        var rects = this.drawPolygon(this.polygons);
		if(scale != null && scale > 0){
			this.polygons.forEach(function(polygon, i) {
				 polygon.rects = rects[i];
			 });
		}
		return rects;
    },

    init : function() {
        //初始获取正五、三、六边形的点位集合
        /*var p1 = this.polygonPoints(5, 100, 100);
        //绘制多边形
        var rects = this.drawPolygon([p1]);
        //初始化多边形数组数据,
        this.polygons.push({
            points: p1,//点位集合
            rects: rects[0],//引脚
            polygonTouch: false,
        });*/
        //右键删除
        this.canvas.oncontextmenu = function(event){
            var event = event || window.event;
            var polygon,polygonIndex;
            polygonThis.polygons.forEach(function(p, i){
                if(polygonThis.pointInPolygon(event.offsetX, event.offsetY, p.points)){
                    polygon = p;
                    polygonIndex = i;
                }
            });
            if("mapMainMonitor" != pageSource && polygon){
                var $li = $("#del_icon").html();
                var $html = '<ul style="list-style: none; padding: 1px; margin: 0px; background-color: rgb(255, 255, 255); border: 1px solid rgb(153, 153, 153); width: auto;">'+
                    '<li id="del_icon" class="door_ops" style="margin: 0px; color: rgb(0, 0, 0); display: block; cursor: default; padding: 3px; border: 1px solid rgb(255, 255, 255); background-color: transparent;">'+
                    $li + '</li></ul>';
                $("#jqContextMenu").next().hide();
                $("#jqContextMenu").css({top:event.clientY + "px", left: event.clientX + "px"}).html($html).show();
                $("#jqContextMenu").find("li").hover(function() {
                  $(this).css({border: "1px solid #0a246a",backgroundColor: "#b6bdd2"});
                }, function() {
                    $(this).css({
                        margin: "0px",
                        color: "#000",
                        display: "block",
                        cursor: "default",
                        padding: "3px",
                        border: "1px solid #fff",
                        backgroundColor: "transparent"
                    });
                }).click(function(){
                    if($(this).attr("id") == "del_icon"){
                        $("#jqContextMenu").hide();
                        $("#jqContextMenu").next().hide();
                        if(window.poPupImageIsMove == false){
                            return;
                        }
                        var itemId = polygon.data.mapElementId;
                        var name =  polygon.data.entityName;
                        var entityId =  polygon.data.entityId;
                        var path = "/mapElement.do?deleteElements&ids="+itemId + "&name=" + encodeURIComponent(name);
                        $.get(path, function(result){
                            dealRetResult(eval(result),function(){
                                $(obj).parent().parent().remove();
                                if(typeof(deleteZtreeNode) == 'function'){
                                    deleteZtreeNode(entityId);
                                }
                                polygonThis.polygons.splice(polygonIndex, 1);
                                polygonThis.repaint();
                            });
                        }, "json");
                    }
                })
            }else{
                $("#jqContextMenu").hide();
                $("#jqContextMenu").next().hide();
            }
            return false;//屏蔽浏览器自带的右键菜单
        };
        //隐藏删除菜单
        this.canvas.onclick = function(e){
            $("#jqContextMenu").hide();
            $("#jqContextMenu").next().hide();
        }
    }
}