/*
 * Decompiled with CFR 0.152.
 */
package jdk.jfr.internal.tool;

import java.io.PrintWriter;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.StringJoiner;
import jdk.jfr.AnnotationElement;
import jdk.jfr.DataAmount;
import jdk.jfr.Frequency;
import jdk.jfr.MemoryAddress;
import jdk.jfr.Percentage;
import jdk.jfr.ValueDescriptor;
import jdk.jfr.consumer.RecordedClass;
import jdk.jfr.consumer.RecordedClassLoader;
import jdk.jfr.consumer.RecordedEvent;
import jdk.jfr.consumer.RecordedFrame;
import jdk.jfr.consumer.RecordedMethod;
import jdk.jfr.consumer.RecordedObject;
import jdk.jfr.consumer.RecordedStackTrace;
import jdk.jfr.consumer.RecordedThread;
import jdk.jfr.internal.PrivateAccess;
import jdk.jfr.internal.Type;
import jdk.jfr.internal.Utils;
import jdk.jfr.internal.tool.EventPrintWriter;

public final class PrettyWriter
extends EventPrintWriter {
    private static final String TYPE_OLD_OBJECT = "jdk.types.OldObject";
    private static final DateTimeFormatter TIME_FORMAT = DateTimeFormatter.ofPattern("HH:mm:ss.SSS");
    private static final Long ZERO = 0L;
    private boolean showIds;
    private RecordedEvent currentEvent;

    public PrettyWriter(PrintWriter destination) {
        super(destination);
    }

    @Override
    protected void print(List<RecordedEvent> events) {
        for (RecordedEvent e : events) {
            this.print(e);
            this.flush(false);
        }
    }

    public void printType(Type t) {
        if (this.showIds) {
            this.print("// id: ");
            this.println(String.valueOf(t.getId()));
        }
        int commentIndex = t.getName().length() + 10;
        String typeName = t.getName();
        int index = typeName.lastIndexOf(".");
        if (index != -1) {
            this.println("@Name(\"" + typeName + "\")");
        }
        this.printAnnotations(commentIndex, t.getAnnotationElements());
        this.print("class " + typeName.substring(index + 1));
        String superType = t.getSuperType();
        if (superType != null) {
            this.print(" extends " + superType);
        }
        this.println(" {");
        this.indent();
        boolean first = true;
        for (ValueDescriptor v : t.getFields()) {
            this.printField(commentIndex, v, first);
            first = false;
        }
        this.retract();
        this.println("}");
        this.println();
    }

    private void printField(int commentIndex, ValueDescriptor v, boolean first) {
        if (!first) {
            this.println();
        }
        this.printAnnotations(commentIndex, v.getAnnotationElements());
        this.printIndent();
        Type vType = PrivateAccess.getInstance().getType(v);
        if (Type.SUPER_TYPE_SETTING.equals(vType.getSuperType())) {
            this.print("static ");
        }
        this.print(this.makeSimpleType(v.getTypeName()));
        if (v.isArray()) {
            this.print("[]");
        }
        this.print(" ");
        this.print(v.getName());
        this.print(";");
        this.printCommentRef(commentIndex, v.getTypeId());
    }

    private void printCommentRef(int commentIndex, long typeId) {
        if (this.showIds) {
            int column = this.getColumn();
            if (column > commentIndex) {
                this.print("  ");
            } else {
                while (column < commentIndex) {
                    this.print(" ");
                    ++column;
                }
            }
            this.println(" // id=" + typeId);
        } else {
            this.println();
        }
    }

    private void printAnnotations(int commentIndex, List<AnnotationElement> annotations) {
        for (AnnotationElement a : annotations) {
            this.printIndent();
            this.print("@");
            this.print(this.makeSimpleType(a.getTypeName()));
            List<ValueDescriptor> vs = a.getValueDescriptors();
            if (!vs.isEmpty()) {
                this.printAnnotation(a);
                this.printCommentRef(commentIndex, a.getTypeId());
                continue;
            }
            this.println();
        }
    }

    private void printAnnotation(AnnotationElement a) {
        StringJoiner sj = new StringJoiner(", ", "(", ")");
        List<ValueDescriptor> vs = a.getValueDescriptors();
        for (ValueDescriptor v : vs) {
            Object o = a.getValue(v.getName());
            if (vs.size() == 1 && v.getName().equals("value")) {
                sj.add(this.textify(o));
                continue;
            }
            sj.add(v.getName() + "=" + this.textify(o));
        }
        this.print(sj.toString());
    }

    private String textify(Object o) {
        if (o.getClass().isArray()) {
            Object[] array = (Object[])o;
            if (array.length == 1) {
                return this.quoteIfNeeded(array[0]);
            }
            StringJoiner s = new StringJoiner(", ", "{", "}");
            for (Object ob : array) {
                s.add(this.quoteIfNeeded(ob));
            }
            return s.toString();
        }
        return this.quoteIfNeeded(o);
    }

    private String quoteIfNeeded(Object o) {
        if (o instanceof String) {
            return "\"" + o + "\"";
        }
        return String.valueOf(o);
    }

    private String makeSimpleType(String typeName) {
        int index = typeName.lastIndexOf(".");
        return typeName.substring(index + 1);
    }

    public void print(RecordedEvent event) {
        this.currentEvent = event;
        this.print(event.getEventType().getName(), " ");
        this.println("{");
        this.indent();
        for (ValueDescriptor v : event.getFields()) {
            String name = v.getName();
            if (this.isZeroDuration(event, name) || this.isLateField(name)) continue;
            this.printFieldValue(event, v);
        }
        if (event.getThread() != null) {
            this.printIndent();
            this.print("eventThread = ");
            this.printThread(event.getThread(), "");
        }
        if (event.getStackTrace() != null) {
            this.printIndent();
            this.print("stackTrace = ");
            this.printStackTrace(event.getStackTrace());
        }
        this.retract();
        this.printIndent();
        this.println("}");
        this.println();
    }

    private boolean isZeroDuration(RecordedEvent event, String name) {
        return name.equals("duration") && ZERO.equals(event.getValue("duration"));
    }

    private void printStackTrace(RecordedStackTrace stackTrace) {
        this.println("[");
        List<RecordedFrame> frames = stackTrace.getFrames();
        this.indent();
        int i = 0;
        while (i < frames.size() && i < this.getStackDepth()) {
            RecordedFrame frame = frames.get(i);
            if (!frame.isJavaFrame()) continue;
            this.printIndent();
            this.printValue(frame, null, "");
            this.println();
            ++i;
        }
        if (stackTrace.isTruncated() || i == this.getStackDepth()) {
            this.printIndent();
            this.println("...");
        }
        this.retract();
        this.printIndent();
        this.println("]");
    }

    public void print(RecordedObject struct, String postFix) {
        this.println("{");
        this.indent();
        for (ValueDescriptor v : struct.getFields()) {
            this.printFieldValue(struct, v);
        }
        this.retract();
        this.printIndent();
        this.println("}" + postFix);
    }

    private void printFieldValue(RecordedObject struct, ValueDescriptor v) {
        this.printIndent();
        this.print(v.getName(), " = ");
        this.printValue(this.getValue(struct, v), v, "");
    }

    private void printArray(Object[] array) {
        this.println("[");
        this.indent();
        for (int i = 0; i < array.length; ++i) {
            this.printIndent();
            this.printValue(array[i], null, i + 1 < array.length ? ", " : "");
        }
        this.retract();
        this.printIndent();
        this.println("]");
    }

    private void printValue(Object value, ValueDescriptor field, String postFix) {
        Integer i;
        Long l;
        Float f;
        Double d;
        if (value == null) {
            this.println("N/A" + postFix);
            return;
        }
        if (value instanceof RecordedObject) {
            RecordedFrame frame;
            if (value instanceof RecordedThread) {
                this.printThread((RecordedThread)value, postFix);
                return;
            }
            if (value instanceof RecordedClass) {
                this.printClass((RecordedClass)value, postFix);
                return;
            }
            if (value instanceof RecordedClassLoader) {
                this.printClassLoader((RecordedClassLoader)value, postFix);
                return;
            }
            if (value instanceof RecordedFrame && (frame = (RecordedFrame)value).isJavaFrame()) {
                this.printJavaFrame((RecordedFrame)value, postFix);
                return;
            }
            if (value instanceof RecordedMethod) {
                this.println(this.formatMethod((RecordedMethod)value));
                return;
            }
            if (field.getTypeName().equals(TYPE_OLD_OBJECT)) {
                this.printOldObject((RecordedObject)value);
                return;
            }
            this.print((RecordedObject)value, postFix);
            return;
        }
        if (value.getClass().isArray()) {
            this.printArray((Object[])value);
            return;
        }
        if (value instanceof Double && (Double.isNaN(d = (Double)value) || d == Double.NEGATIVE_INFINITY)) {
            this.println("N/A");
            return;
        }
        if (value instanceof Float && (Float.isNaN((f = (Float)value).floatValue()) || f.floatValue() == Float.NEGATIVE_INFINITY)) {
            this.println("N/A");
            return;
        }
        if (value instanceof Long && (l = (Long)value) == Long.MIN_VALUE) {
            this.println("N/A");
            return;
        }
        if (value instanceof Integer && (i = (Integer)value) == Integer.MIN_VALUE) {
            this.println("N/A");
            return;
        }
        if (field.getContentType() != null && this.printFormatted(field, value)) {
            return;
        }
        String text = String.valueOf(value);
        if (value instanceof String) {
            text = "\"" + text + "\"";
        }
        this.println(text);
    }

    private void printOldObject(RecordedObject object) {
        this.println(" [");
        this.indent();
        this.printIndent();
        try {
            this.printReferenceChain(object);
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        this.retract();
        this.printIndent();
        this.println("]");
    }

    private void printReferenceChain(RecordedObject object) {
        this.printObject(object, this.currentEvent.getLong("arrayElements"));
        RecordedObject ref = (RecordedObject)object.getValue("referrer");
        while (ref != null) {
            RecordedObject field;
            long skip = ref.getLong("skip");
            if (skip > 0L) {
                this.printIndent();
                this.println("...");
            }
            String objectHolder = "";
            long size = Long.MIN_VALUE;
            RecordedObject array = (RecordedObject)ref.getValue("array");
            if (array != null) {
                long index = array.getLong("index");
                size = array.getLong("size");
                objectHolder = "[" + index + "]";
            }
            if ((field = (RecordedObject)ref.getValue("field")) != null) {
                objectHolder = field.getString("name");
            }
            this.printIndent();
            this.print(objectHolder);
            this.print(" : ");
            object = (RecordedObject)ref.getValue("object");
            if (object != null) {
                this.printObject(object, size);
            }
            ref = (RecordedObject)object.getValue("referrer");
        }
    }

    void printObject(RecordedObject object, long arraySize) {
        RecordedClass clazz = object.getClass("type");
        if (clazz != null) {
            String className = clazz.getName();
            if (className != null && className.startsWith("[")) {
                className = this.decodeDescriptors(className, arraySize > 0L ? Long.toString(arraySize) : "").get(0);
            }
            this.print(className);
            String description = object.getString("description");
            if (description != null) {
                this.print(" ");
                this.print(description);
            }
        }
        this.println();
    }

    private void printClassLoader(RecordedClassLoader cl, String postFix) {
        RecordedClass clazz = cl.getType();
        this.print(clazz == null ? "null" : clazz.getName());
        if (clazz != null) {
            this.print(" (");
            this.print("id = ");
            this.print(String.valueOf(cl.getId()));
            this.println(")");
        }
    }

    private void printJavaFrame(RecordedFrame f, String postFix) {
        this.print(this.formatMethod(f.getMethod()));
        int line = f.getLineNumber();
        if (line >= 0) {
            this.print(" line: " + line);
        }
        this.print(postFix);
    }

    private String formatMethod(RecordedMethod m) {
        StringBuilder sb = new StringBuilder();
        sb.append(m.getType().getName());
        sb.append(".");
        sb.append(m.getName());
        sb.append("(");
        StringJoiner sj = new StringJoiner(", ");
        String md = m.getDescriptor().replace("/", ".");
        String parameter = md.substring(1, md.lastIndexOf(")"));
        for (String qualifiedName : this.decodeDescriptors(parameter, "")) {
            String typeName = qualifiedName.substring(qualifiedName.lastIndexOf(46) + 1);
            sj.add(typeName);
        }
        sb.append(sj);
        sb.append(")");
        return sb.toString();
    }

    private void printClass(RecordedClass clazz, String postFix) {
        String className;
        RecordedClassLoader classLoader = clazz.getClassLoader();
        String classLoaderName = "null";
        if (classLoader != null) {
            classLoaderName = classLoader.getName() != null ? classLoader.getName() : classLoader.getType().getName();
        }
        if ((className = clazz.getName()).startsWith("[")) {
            className = this.decodeDescriptors(className, "").get(0);
        }
        this.println(className + " (classLoader = " + classLoaderName + ")" + postFix);
    }

    List<String> decodeDescriptors(String descriptor, String arraySize) {
        ArrayList<String> descriptors = new ArrayList<String>();
        for (int index = 0; index < descriptor.length(); ++index) {
            String type;
            String arrayBrackets = "";
            while (descriptor.charAt(index) == '[') {
                arrayBrackets = arrayBrackets + "[" + arraySize + "]";
                arraySize = "";
                ++index;
            }
            char c = descriptor.charAt(index);
            switch (c) {
                case 'L': {
                    int endIndex = descriptor.indexOf(59, index);
                    type = descriptor.substring(index + 1, endIndex);
                    index = endIndex;
                    break;
                }
                case 'I': {
                    type = "int";
                    break;
                }
                case 'J': {
                    type = "long";
                    break;
                }
                case 'Z': {
                    type = "boolean";
                    break;
                }
                case 'D': {
                    type = "double";
                    break;
                }
                case 'F': {
                    type = "float";
                    break;
                }
                case 'S': {
                    type = "short";
                    break;
                }
                case 'C': {
                    type = "char";
                    break;
                }
                case 'B': {
                    type = "byte";
                    break;
                }
                default: {
                    type = "<unknown-descriptor-type>";
                }
            }
            descriptors.add(type + arrayBrackets);
        }
        return descriptors;
    }

    private void printThread(RecordedThread thread, String postFix) {
        long javaThreadId = thread.getJavaThreadId();
        if (javaThreadId > 0L) {
            this.println("\"" + thread.getJavaName() + "\" (javaThreadId = " + thread.getJavaThreadId() + ")" + postFix);
        } else {
            this.println("\"" + thread.getOSName() + "\" (osThreadId = " + thread.getOSThreadId() + ")" + postFix);
        }
    }

    private boolean printFormatted(ValueDescriptor field, Object value) {
        MemoryAddress memoryAddress;
        if (value instanceof Duration) {
            Duration d = (Duration)value;
            if (d.getSeconds() == Long.MIN_VALUE && d.getNano() == 0) {
                this.println("N/A");
                return true;
            }
            double s = (double)d.getNano() / 1.0E9 + (double)((int)(d.getSeconds() % 60L));
            if (s < 1.0) {
                if (s < 0.001) {
                    this.println(String.format("%.3f", s * 1000000.0) + " us");
                } else {
                    this.println(String.format("%.3f", s * 1000.0) + " ms");
                }
            } else if (s < 1000.0) {
                this.println(String.format("%.3f", s) + " s");
            } else {
                this.println(String.format("%.0f", s) + " s");
            }
            return true;
        }
        if (value instanceof OffsetDateTime) {
            OffsetDateTime odt = (OffsetDateTime)value;
            if (odt.equals(OffsetDateTime.MIN)) {
                this.println("N/A");
                return true;
            }
            this.println(TIME_FORMAT.format(odt));
            return true;
        }
        Percentage percentage = field.getAnnotation(Percentage.class);
        if (percentage != null && value instanceof Number) {
            double d = ((Number)value).doubleValue();
            this.println(String.format("%.2f", d * 100.0) + "%");
            return true;
        }
        DataAmount dataAmount = field.getAnnotation(DataAmount.class);
        if (dataAmount != null && value instanceof Number) {
            Number n = (Number)value;
            long amount = n.longValue();
            if (field.getAnnotation(Frequency.class) != null) {
                if (dataAmount.value().equals("BYTES")) {
                    this.println(Utils.formatBytesPerSecond(amount));
                    return true;
                }
                if (dataAmount.value().equals("BITS")) {
                    this.println(Utils.formatBitsPerSecond(amount));
                    return true;
                }
            } else {
                if (dataAmount.value().equals("BYTES")) {
                    this.println(Utils.formatBytes(amount));
                    return true;
                }
                if (dataAmount.value().equals("BITS")) {
                    this.println(Utils.formatBits(amount));
                    return true;
                }
            }
        }
        if ((memoryAddress = field.getAnnotation(MemoryAddress.class)) != null && value instanceof Number) {
            long d = ((Number)value).longValue();
            this.println(String.format("0x%08X", d));
            return true;
        }
        Frequency frequency = field.getAnnotation(Frequency.class);
        if (frequency != null && value instanceof Number) {
            this.println(value + " Hz");
            return true;
        }
        return false;
    }

    public void setShowIds(boolean showIds) {
        this.showIds = showIds;
    }
}

